<?php

namespace App\Http\Controllers;

use App\Http\Requests\MemberRequest;
use App\Models\Member;
use App\Models\MembershipType;
use App\Models\Membership;
use Auth;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;

class MemberController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->has('search')) {
            $keyword = $request->input('search');
            $paginate = false;
            $members = Member::where('name', 'LIKE', '%' . $keyword . '%')
                ->orWhere('reg_no', 'LIKE', '%' . $keyword . '%')
                ->orWhere('phone', 'LIKE', '%' . $keyword . '%')
                ->with('latest_membership.membership_type')
                ->orderBy('id', 'DESC')->get();

            return view('member.list', compact('members', 'paginate'));
        } else {
            $paginate = true;
            $members = Member::with('latest_membership.membership_type')->orderBy('id', 'DESC')->paginate(10)->onEachSide(0);
            return view('member.list', compact('members', 'paginate'));
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $types = MembershipType::get();
        $reg_no = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
        return view('member.create', compact('types', 'reg_no'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(MemberRequest $request)
    {
        $membership_type_id = $request->input('type');
        $profile_picture = $this->upload_image($request, "");
        $membership_type = MembershipType::find($membership_type_id);

        $member = new Member();
        $member->name = $request->input('name');
        $member->reg_no = $request->input('reg_no');
        $member->profile_picture = $profile_picture;
        $member->interested_area = $request->input('interested_area');
        $member->gender = $request->input('gender');
        $member->age = $request->input('age');
        $member->address = $request->input('address');
        $member->phone = $request->input('phone');
        $member->height = $request->input('height');
        $member->weight = $request->input('weight');
        $member->status = $request->input('status');
        $member->save();

        $membership = $member->memberships()->create([
            'membership_type_id' => $membership_type->id,
            'valid_from' => $request->input('valid_from'),
            'valid_to' => $request->input('valid_to')
        ]);

        $membership->transactions()->create([
            'amount' => $membership_type->amount,
            'fee' => 0,
            'dr_cr' => 'dr',
            'date' => Carbon::now(),
            'method' => 'online',
            'note' => $membership_type->name . ' Membership fee',
            'created_user_id' => Auth::user()->id,
            'type' => 'membership fee'
        ]);

        return back()->with('success', 'Saved successfully');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Member $member)
    {
        $member = Member::with('latest_membership.membership_type', 'memberships')->find($member->id);
        $types = MembershipType::get();

        return view('member.edit', compact('member', 'types'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(MemberRequest $request, Member $member)
    {
        $membership_type_id = $request->input('type');
        $membership_type = MembershipType::find($membership_type_id);
        $member = Member::find($member->id);

        $profile_picture = $this->upload_image($request, $member->profile_picture);
        
        $member->name = $request->input('name');
        $member->reg_no = $request->input('reg_no');
        $member->profile_picture = $profile_picture;
        $member->status = $request->input('status');
        $member->interested_area = $request->input('interested_area');
        $member->gender = $request->input('gender');
        $member->age = $request->input('age');
        $member->address = $request->input('address');
        $member->phone = $request->input('phone');
        $member->height = $request->input('height');
        $member->weight = $request->input('weight');

        $member->save();

        $membership = Membership::updateOrCreate(
            ['id' => $request->input('type')],
            [
                'member_id' => $member->id,
                'membership_type_id' => $request->input('type'),
                'valid_from' => $request->input('valid_from'),
                'valid_to' => $request->input('valid_to')
            ]
        );

        $membership->transactions()->create([
            'amount' => $membership_type->amount,
            'fee' => 0,
            'dr_cr' => 'dr',
            'date' => Carbon::now(),
            'method' => 'online',
            'note' => 'New membership assigned to the' . ' ' . $request->input('name'),
            'created_user_id' => Auth::user()->id,
            'type' => 'membership'
        ]);

        return redirect()->route('member.index')->with('success', 'Updated successfully');
    }

    /**
     * Remove a member and associated data from the database.
     *
     * @param  \App\Models\Member  $member  The member to be deleted.
     * @return \Illuminate\Http\RedirectResponse  A redirection to the member index page with a success message.
     */
    public function destroy(Member $member)
    {
        // code for remove old file
        if ($member->image != '' && $member->image != null) {
            $this->remove_image($member->image);
        }

        $memberships = $member->memberships;
        foreach ($memberships as $membership) {
            $transactions = $membership->transactions;
            foreach ($transactions as $transaction) {
                $transaction->delete();
            }
            $membership->delete();
        }

        $attendances = $member->attendances;
        foreach ($attendances as $attendance) {
            $attendance->delete();
        }
        
        $member->delete();
        return redirect()->route('member.index')->with('success', 'Deleted successfully');
    }

    // GET THE LIST OF UNPAID MEMBERS
    public function unpaid(Request $request)
    {
        $query = DB::table('members')
            ->join('memberships', 'members.id', '=', 'memberships.member_id')
            ->leftJoin('membership_types', 'memberships.membership_type_id', '=', 'membership_types.id')
            ->leftJoin('transactions', 'memberships.id', '=', 'transactions.transactionable_id')
            ->select(
                'members.reg_no',
                'members.name',
                'members.phone',
                'memberships.valid_from',
                'memberships.valid_to',
                'memberships.id',
                'membership_types.name AS membership_name',
                DB::raw('COALESCE(SUM(CASE WHEN transactions.dr_cr = "dr" THEN transactions.amount ELSE 0 END), 0) AS actual_amount'),
                DB::raw('COALESCE(SUM(CASE WHEN transactions.dr_cr = "cr" THEN transactions.amount ELSE 0 END), 0) AS total_paid'),
                DB::raw('COALESCE(SUM(CASE WHEN transactions.dr_cr = "dr" THEN transactions.amount ELSE 0 END), 0) - COALESCE(SUM(CASE WHEN transactions.dr_cr = "cr" THEN transactions.amount ELSE 0 END), 0) AS due_amount')
            )
            ->groupBy(
                'members.reg_no',
                'members.name',
                'members.phone',
                'memberships.valid_from',
                'memberships.valid_to',
                'memberships.id',
                'membership_name'
            )
            ->havingRaw('total_paid < actual_amount');

        if ($request->has('search')) {
            $keyword = $request->input('search');
            $paginate = false;

            $memberships = $query->where('members.reg_no', 'LIKE', '%' . $keyword . '%')
                ->orWhere('members.name', 'LIKE', '%' . $keyword . '%')
                ->orWhere('members.phone', 'LIKE', '%' . $keyword . '%')
                ->orWhere('membership_types.name', 'LIKE', '%' . $keyword . '%')
                ->orderBy('memberships.created_at', 'DESC')
                ->get();

            return view('alert.unpaid_members', compact('memberships', 'paginate'));
        } else {
            $paginate = true;
            $memberships = $query->orderBy('memberships.created_at', 'DESC')->paginate(10)->onEachSide(0);

            return view('alert.unpaid_members', compact('memberships', 'paginate'));
        }
    }

    // GET THE LIST OF EXPIRED MEMBERSHIP MEMBERS
    public function expired(Request $request)
    {
        $query = DB::table('members')
            ->join('memberships', 'members.id', '=', 'memberships.member_id')
            ->leftJoin('membership_types', 'memberships.membership_type_id', '=', 'membership_types.id')
            ->leftJoin('transactions', 'memberships.id', '=', 'transactions.transactionable_id')
            ->where('memberships.created_at', function ($query) {
                $query->select('created_at')
                    ->from('memberships')
                    ->whereColumn('member_id', 'members.id')
                    ->orderByDesc('created_at')
                    ->limit(1);
            })
            ->where('memberships.valid_to', '<', now())
            ->select(
                'members.id AS member_id',
                'members.reg_no',
                'members.name',
                'members.phone',
                'memberships.valid_from',
                'memberships.valid_to',
                'memberships.id',
                'membership_types.name AS membership_name',
                DB::raw('COALESCE(SUM(CASE WHEN transactions.dr_cr = "dr" THEN transactions.amount ELSE 0 END), 0) AS actual_amount'),
                DB::raw('COALESCE(SUM(CASE WHEN transactions.dr_cr = "cr" THEN transactions.amount ELSE 0 END), 0) AS total_paid'),
                DB::raw('COALESCE(SUM(CASE WHEN transactions.dr_cr = "dr" THEN transactions.amount ELSE 0 END), 0) - COALESCE(SUM(CASE WHEN transactions.dr_cr = "cr" THEN transactions.amount ELSE 0 END), 0) AS due_amount')
            )
            ->groupBy(
                'members.id',
                'members.reg_no',
                'members.name',
                'members.phone',
                'memberships.valid_from',
                'memberships.valid_to',
                'memberships.id',
                'membership_types.name'
            )
            ->orderByDesc('memberships.valid_to');

        if ($request->has('search')) {
            $keyword = $request->input('search');
            $paginate = false;

            $memberships = $query->where(function ($q) use ($keyword) {
                $q->where('members.reg_no', 'LIKE', '%' . $keyword . '%')
                    ->orWhere('members.name', 'LIKE', '%' . $keyword . '%')
                    ->orWhere('members.phone', 'LIKE', '%' . $keyword . '%')
                    ->orWhere('membership_types.name', 'LIKE', '%' . $keyword . '%');
            })->orderBy('memberships.created_at', 'DESC')->get();

            return view('alert.expired_members', compact('memberships', 'paginate'));
        } else {
            $paginate = true;
            $memberships = $query->orderBy('memberships.created_at', 'DESC')->paginate(10)->onEachSide(0);

            return view('alert.expired_members', compact('memberships', 'paginate'));
        }
    }

    // UPLOAD THE NEW IMAGE
    private function upload_image(Request $request, $name)
    {
        if ($request->hasfile('profile_picture')) {
            if (!empty($name)) {
                $this->remove_image($name);
            }
            $file = $request->file('profile_picture');
            $name = unique_image_name($file->getClientOriginalExtension());
            $file->move('images/profile', $name);
        }

        return $name;
    }

    // REMOVE THE OLD IMAGE FROM THE STORAGE
    private function remove_image($name)
    {
        if (!empty($name)) {
            $path = 'images/profile/';
            $file_old = $path . $name;
            if (file_exists($file_old)) {
                unlink($file_old);
            }
        }
    }
}